<?php
/**
 * Wishlists handler premium class
 *
 * @package YITH\Wishlist\Classes
 * @author  YITH <plugins@yithemes.com>
 * @since   4.0.0
 */

defined( 'YITH_WCWL' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WCWL_Wishlists_Premium' ) ) {
	/**
	 * YITH_WCWL_Wishlist class
	 */
	class YITH_WCWL_Wishlists_Premium extends YITH_WCWL_Wishlists {
		/* === WISHLISTS METHODS === */

		/**
		 * Add a new wishlist for the user.
		 *
		 * @param array $args Add to Wishlist info:
		 *                    [
		 *                    'wishlist_name'
		 *                    'wishlist_visibility'
		 *                    'user_id'
		 *                    'session_id'
		 *                    ].
		 *
		 * @throws YITH_WCWL_Exception When something goes wrong with creation.
		 * @return YITH_WCWL_Wishlist
		 * @since 3.0.0
		 */
		public function create( $args = array() ) {
			$defaults = array(
				'wishlist_name'       => false,
				'wishlist_visibility' => 0,
				'user_id'             => false,
				'session_id'          => false,
			);

			$args = apply_filters( 'yith_wcwl_adding_wishlist_args', wp_parse_args( $args, $defaults ) );

			$wishlist_name       = sanitize_text_field( $args[ 'wishlist_name' ] );
			$wishlist_slug       = sanitize_title_with_dashes( $args[ 'wishlist_name' ] );
			$session_id          = sanitize_title_with_dashes( $args[ 'session_id' ] );
			$wishlist_visibility = intval( $args[ 'wishlist_visibility' ] );
			$user_id             = intval( $args[ 'user_id' ] );

			if ( $user_id && $user_id !== get_current_user_id() && !current_user_can( 'manage_woocommerce' ) ) {
				throw new YITH_WCWL_Exception( esc_html__( 'There was an error while processing your request; please, try later', 'yith-woocommerce-wishlist' ), 0 );
			}

			if ( ! $wishlist_name ) {
				throw new YITH_WCWL_Exception( esc_html__( 'Wishlist name is required', 'yith-woocommerce-wishlist' ), 0 );
			} elseif ( strlen( $wishlist_name ) >= 65535 ) {
				throw new YITH_WCWL_Exception( esc_html__( 'Wishlist name exceeds the maximum number of characters allowed', 'yith-woocommerce-wishlist' ), 0 );
			}

			$new_wishlist = new YITH_WCWL_Wishlist();

			// set properties before saving.
			$new_wishlist->set_slug( $wishlist_slug );
			$new_wishlist->set_name( $wishlist_name );
			$new_wishlist->set_privacy( $wishlist_visibility );
			$new_wishlist->set_user_id( $user_id );
			$new_wishlist->set_session_id( $session_id );

			$new_wishlist->save();

			/**
			 * DO_ACTION: yith_wcwl_after_add_wishlist
			 *
			 * Allows to fire some action when a new wishlist has been created.
			 *
			 * @param YITH_WCWL_Wishlist $new_wishlist New wishlist object
			 */
			do_action( 'yith_wcwl_after_add_wishlist', $new_wishlist );

			return $new_wishlist;
		}

		/**
		 * Update wishlist with arguments passed as second parameter
		 *
		 * @param int   $wishlist_id Wishlist id.
		 * @param array $args        Array of parameters to use for update query.
		 *
		 * @throws YITH_WCWL_Exception When something goes wrong with update.
		 *
		 * @return void
		 */
		public function update( $wishlist_id, $args = array() ) {
			$wishlist = YITH_WCWL_Wishlist_Factory::get_wishlist( $wishlist_id );

			if ( ! $wishlist ) {
				throw new YITH_WCWL_Exception( esc_html__( 'Couldn\'t find any wishlist with the provided ID', 'yith-woocommerce-wishlist' ), 0 );
			}

			if ( ! $wishlist->current_user_can( 'update_wishlist' ) ) {
				throw new YITH_WCWL_Exception( esc_html__( 'There was an error while processing your request; please, try later', 'yith-woocommerce-wishlist' ), 0 ); // @since 3.0.7
			}

			if ( isset( $args[ 'wishlist_name' ] ) ) {
				$wishlist_name = $args[ 'wishlist_name' ];

				if ( ! $wishlist_name ) {
					throw new YITH_WCWL_Exception( esc_html__( 'Wishlist name is required', 'yith-woocommerce-wishlist' ), 0 );
				} elseif ( strlen( $wishlist_name ) >= 65535 ) {
					throw new YITH_WCWL_Exception( esc_html__( 'Wishlist name exceeds the maximum number of characters allowed', 'yith-woocommerce-wishlist' ), 0 );
				}

				$wishlist->set_name( $args[ 'wishlist_name' ] );
			}

			if ( isset( $args[ 'wishlist_visibility' ] ) || isset( $args[ 'wishlist_privacy' ] ) ) {
				$wishlist->set_privacy( isset( $args[ 'wishlist_visibility' ] ) ? $args[ 'wishlist_visibility' ] : $args[ 'wishlist_privacy' ] );
			}

			$wishlist->save();
		}

		/**
		 * Delete indicated wishlist
		 *
		 * @param int $wishlist_id Wishlist id.
		 * @throws YITH_WCWL_Exception When something goes wrong with deletion.
		 * @return void
		 * @since 3.0.0
		 */
		public function remove( $wishlist_id ) {
			$wishlist = YITH_WCWL_Wishlist_Factory::get_wishlist( $wishlist_id );

			if ( ! $wishlist ) {
				throw new YITH_WCWL_Exception( esc_html__( 'Couldn\'t find any wishlist with the provided ID', 'yith-woocommerce-wishlist' ), 0 );
			}

			if ( ! $wishlist->current_user_can( 'remove_wishlist' ) ) {
				throw new YITH_WCWL_Exception( esc_html__( 'There was an error while processing your request; please, try later', 'yith-woocommerce-wishlist' ), 0 ); // @since 3.0.7
			}

			$wishlist->delete();
		}

		/**
		 * Move an item from a wishlist to another
		 *
		 * @param array $args The arguments
		 * @return array
		 *
		 * @throws YITH_WCWL_Exception
		 */
		public function move( $args ) {
			if ( ! isset( $args[ 'destination_wishlist' ] ) ) throw new YITH_WCWL_Exception( esc_html__( 'You must provide a destination wishlist', 'yith-woocommerce-wishlist' ) );

			$origin_wishlist = YITH_WCWL_Wishlist_Factory::get_wishlist( $args[ 'origin_wishlist' ] );
			$product         = wc_get_product( $args[ 'product_id' ] );

			if ( ! $product ) throw new YITH_WCWL_Exception( esc_html__( 'You must provide a valid product', 'yith-woocommerce-wishlist' ) );
			if ( ! $origin_wishlist ) throw new YITH_WCWL_Exception( esc_html__( 'Unable to retrieve the origin wishlist', 'yith-woocommerce-wishlist' ) );

			$item    = $this->get_product_item( $product->get_id(), $origin_wishlist->get_id() );
			$item_id = $item ? $item->get_id() : false;
			$moved   = false;

			if ( 'new' === $args[ 'destination_wishlist' ] ) {
				if ( ! isset( $args[ 'wishlist_name' ], $args[ 'wishlist_visibility' ] ) ) {
					throw new YITH_WCWL_Exception( esc_html__( 'You must provide a destination wishlist name and visibility', 'yith-woocommerce-wishlist' ) );
				}

				$destination_wishlist = $this->create( array_intersect_key( $args, array_flip( array( 'wishlist_name', 'wishlist_visibility' ) ) ) );
			} else {
				$destination_wishlist = YITH_WCWL_Wishlist_Factory::get_wishlist( $args[ 'destination_wishlist' ] );
			}

			if ( ! $destination_wishlist ) throw new YITH_WCWL_Exception( esc_html__( 'Unable to retrieve the destination wishlist', 'yith-woocommerce-wishlist' ) );

			if ( $origin_wishlist && $destination_wishlist && $origin_wishlist->current_user_can( 'remove_from_wishlist' ) && $destination_wishlist->current_user_can( 'add_to_wishlist' ) ) {
				if ( $item ) {
					$destination_item = $destination_wishlist->get_product( $item_id );

					if ( $destination_item ) {
						$destination_item->set_date_added( current_time( 'mysql' ) );

						$destination_item->save();
						$item->delete();
					} else {
						$item->set_wishlist_id( $destination_wishlist->get_id() );
						$item->set_date_added( current_time( 'mysql' ) );

						$item->save();
					}

					$moved = true;
					wp_cache_delete( 'wishlist-items-' . $origin_wishlist->get_id(), 'wishlists' );
					wp_cache_delete( 'wishlist-items-' . $destination_wishlist->get_id(), 'wishlists' );

				}
			}

			return compact( 'moved', 'destination_wishlist' );
		}

		/* === GENERAL METHODS === */

		/**
		 * Checks whether multi-wishlist feature is enabled for current user
		 *
		 * @return bool Whether feature is enabled or not
		 */
		public function is_multi_wishlist_enabled() {
			$multi_wishlist_enabled = 'yes' === get_option( 'yith_wcwl_multi_wishlist_enable', 'no' );

			if ( $multi_wishlist_enabled && ! is_user_logged_in() ) {
				$multi_wishlist_enabled = ( 'yes' === get_option( 'yith_wcwl_enable_multi_wishlist_for_unauthenticated_users', 'no' ) );
			}

			/**
			 * APPLY_FILTERS: yith_wcwl_is_wishlist_enabled
			 *
			 * Filter whether the multi-wishlist is enabled for current user.
			 *
			 * @param bool $multi_wishlist_enabled Whether multi-wishlist is enabled or not
			 *
			 * @return bool
			 */
			return apply_filters( 'yith_wcwl_is_wishlist_enabled', $multi_wishlist_enabled );
		}
	}
}
