<?php
/**
 * Init premium features of the plugin
 *
 * @package YITH\Wishlist\Classes
 * @author  YITH <plugins@yithemes.com>
 * @version 3.0.0
 */

defined( 'YITH_WCWL_PREMIUM' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WCWL_Premium' ) ) {
	/**
	 * WooCommerce Wishlist Premium
	 *
	 * @since 1.0.0
	 */
	class YITH_WCWL_Premium extends YITH_WCWL_Premium_Legacy {
		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			parent::__construct();

			// register plugin to licence/update system.
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_activation' ), 99 );
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_updates' ), 99 );

			// on sale item handling.
			add_action( 'yith_wcwl_item_is_on_sale', array( $this, 'schedule_on_sale_item_emails' ), 10, 1 );
		}

		/**
		 * Init an array of plugin emails
		 *
		 * @author Francesco Licandro
		 * @since  1.5.0
		 */
		public function init_plugin_emails_array() {
			$this->emails = array(
				'yith_wcwl_back_in_stock',
				'yith_wcwl_on_sale_item',
				'estimate_mail',
				'yith_wcwl_promotion_mail',
			);
		}

		/* === ITEM ON SALE HANDLING === */

		/**
		 * Schedule on sale item email notification when an item switches to on sale
		 *
		 * @param \YITH_WCWL_Wishlist_Item $item Item on sale.
		 * @return void
		 */
		public function schedule_on_sale_item_emails( $item ) {
			$product_id = $item->get_product_id();
			$product    = $item->get_product();
			$user_id    = $item->get_user_id();
			$user       = $item->get_user();

			if ( ! $user ) {
				return;
			}

			// skip if email ain't active.
			$email_options = get_option( 'woocommerce_yith_wcwl_on_sale_item_settings', array() );

			if ( ! isset( $email_options['enabled'] ) || 'yes' !== $email_options['enabled'] ) {
				return;
			}

			// skip if product is on exclusion list.
			$product_exclusions = ! empty( $email_options['product_exclusions'] ) ? array_map( 'absint', $email_options['product_exclusions'] ) : false;

			if ( $product_exclusions && in_array( $product_id, $product_exclusions, true ) ) {
				return;
			}

			// skip if product category is on exclusion list.
			$product_categories = $product->get_category_ids();

			if ( ! empty( $email_options['category_exclusions'] ) && array_intersect( $product_categories, $email_options['category_exclusions'] ) ) {
				return;
			}

			// queue handling.
			$queue        = get_option( 'yith_wcwl_on_sale_item_queue', array() );
			$unsubscribed = get_option( 'yith_wcwl_unsubscribed_users', array() );

			// skip if user unsubscribed.
			if ( in_array( $user->user_email, $unsubscribed, true ) ) {
				return;
			}

			if ( ! isset( $queue[ $user_id ] ) ) {
				$queue[ $user_id ] = array(
					$item->get_product_id() => $item->get_id(),
				);
			} else {
				$queue[ $user_id ][ $item->get_product_id() ] = $item->get_id();
			}

			update_option( 'yith_wcwl_on_sale_item_queue', $queue );
		}

		/* === GENERAL METHODS === */

		/**
		 * Get current endpoint, if any
		 *
		 * @return string Current endpoint, empty string if no endpoint is being visited
		 */
		public function get_current_endpoint() {
			$action_params = get_query_var( YITH_WCWL()->wishlist_param, false );

			/**
			 * APPLY_FILTERS: yith_wcwl_current_wishlist_view_params
			 *
			 * Filter the array of parameters to see the current wishlist.
			 *
			 * @param array $params Array of parameters
			 *
			 * @return array
			 */
			$action_params    = explode( '/', apply_filters( 'yith_wcwl_current_wishlist_view_params', $action_params ) );
			$current_endpoint = ( isset( $action_params[0] ) ) ? $action_params[0] : '';

			/**
			 * APPLY_FILTERS: yith_wcwl_current_endpoint
			 *
			 * Filter the current endpoint, if any.
			 *
			 * @param string $current_endpoint Current endpoint
			 * @param array  $action_params    Array of parameters
			 *
			 * @return string
			 */
			return apply_filters( 'yith_wcwl_current_endpoint', $current_endpoint, $action_params );
		}

		/**
		 * Check if we're on a specific endpoint
		 *
		 * @param string $endpoint Endpoint to test.
		 * @return bool Whether we're on test endpoint or not
		 */
		public function is_endpoint( $endpoint ) {
			$current_endpoint = $this->get_current_endpoint();

			if ( 'view' === $endpoint && '' === $current_endpoint && yith_wcwl_is_wishlist_page() ) {
				$is_endpoint = true;
			} else {
				$is_endpoint = $current_endpoint === $endpoint;
			}

			/**
			 * APPLY_FILTERS: yith_wcwl_is_endpoint
			 *
			 * Filter whether a specific endpoint is visited.
			 *
			 * @param bool   $is_endpoint      Whether a specific endpoint is visited or not
			 * @param string $endpoint         Endpoint
			 * @param string $current_endpoint Current endpoint
			 *
			 * @return bool
			 */
			return apply_filters( 'yith_wcwl_is_endpoint', $is_endpoint, $endpoint, $current_endpoint );
		}

		/* === WISHLIST LICENCE HANDLING === */

		/**
		 * Register plugins for activation tab
		 *
		 * @return void
		 * @since 2.0.0
		 */
		public function register_plugin_for_activation() {
			if ( ! class_exists( 'YIT_Plugin_Licence' ) ) {
				require_once YITH_WCWL_DIR . 'plugin-fw/licence/lib/yit-licence.php';
				require_once YITH_WCWL_DIR . 'plugin-fw/licence/lib/yit-plugin-licence.php';
			}

			YIT_Plugin_Licence()->register( YITH_WCWL_INIT, YITH_WCWL_SECRET_KEY, YITH_WCWL_SLUG );
		}

		/**
		 * Register plugins for update tab
		 *
		 * @return void
		 * @since 2.0.0
		 */
		public function register_plugin_for_updates() {
			if ( ! class_exists( 'YIT_Upgrade' ) ) {
				require_once YITH_WCWL_DIR . 'plugin-fw/lib/yit-upgrade.php';
			}

			YIT_Upgrade()->register( YITH_WCWL_SLUG, YITH_WCWL_INIT );
		}
	}
}
