<?php
/**
 * Add to wishlist button handler
 *
 * @package YITH\Wishlist\Classes
 * @author  YITH <plugins@yithemes.com>
 * @since   4.0
 */

defined( 'YITH_WCWL' ) || exit; // Exit if accessed directly

if ( ! class_exists( 'YITH_WCWL_Add_To_Wishlist_Button_Premium' ) ) {
	/**
	 * @class YITH_WCWL_Add_To_Wishlist_Button
	 */
	class YITH_WCWL_Add_To_Wishlist_Button_Premium extends YITH_WCWL_Add_To_Wishlist_Button {
		protected static $options = array(
			'modal_enabled'                      => array(
				'name' => 'yith_wcwl_modal_enable',
			),
			'show_count'                         => array(
				'name' => 'yith_wcwl_show_counter',
			),
			'show_count_in_loop'                 => array(
				'name' => 'yith_wcwl_show_counter_in_loop',
			),
			'move_to_another_wishlist'           => array(
				'name' => 'yith_wcwl_show_move_to_another_wishlist',
			),
			'move_to_another_wishlist_type'      => array(
				'name' => 'yith_wcwl_move_to_another_wishlist_type',
			),
			'modal_label'                        => array(
				'name' => 'yith_wcwl_add_to_wishlist_popup_text',
			),
			'disabled_wishlist_for_unauth_users' => array(
				'name' => 'yith_wcwl_disable_wishlist_for_unauthenticated_users',
			),
			'position_over_image_in_loop'        => array(
				'name'    => 'yith_wcwl_position_over_image_in_loop',
				'default' => 'top-left',
			),
			'button_over_image_style'            => array(
				'name'    => 'yith_wcwl_button_over_image_style',
				'default' => 'fixed',
			),
			'feedback_type'                      => array(
				'name'    => 'yith_wcwl_add_to_wishlist_feedback',
				'default' => 'notice',
			),
			'modal_icon_type'                    => array(
				'name'    => 'yith_wcwl_modal_icon_type',
				'default' => 'default',
			),
			'modal_icon'                         => array(
				'name' => 'yith_wcwl_modal_icon',
			),
			'modal_custom_icon'                  => array(
				'name' => 'yith_wcwl_modal_custom_icon',
			),
			'tooltip_enabled'                    => array(
				'name' => 'yith_wcwl_tooltip_enable',
			),
			'modal_close_behaviour'              => array(
				'name' => 'yith_wcwl_modal_close_behaviour',
			),
		);


		/**
		 * the object data
		 *
		 * @var array
		 */
		protected $data = array(
			'product_id'        => false,
			'parent_product_id' => false,
			'icon'              => '',
			'added_icon'        => '',
			'custom_icon'       => '',
			'custom_added_icon' => '',
			'wishlist'          => false,
			'exists'            => false,
			'label'             => '',
			'added_label'       => '',
			'browse_label'      => '',
			'already_in_label'  => '',
			'is_single'         => null,
			'ajax_loading'      => null,
			// PREMIUM DATA.
			'show_count'        => null,
		);

		/* === SETTER === */

		/**
		 * Setter for 'show_count' property
		 *
		 * @param bool $show_count Whether to show the counter
		 */
		public function set_show_count( $show_count ) {
			$this->set_prop( 'show_count', ! ! $show_count );
		}

		/**
		 * Setter for 'show_count' property
		 *
		 * @param bool $show_exists Whether to show the counter
		 */
		public function set_show_exists( $show_exists ) {
			$this->set_prop( 'show_exists', ! ! $show_exists );
		}

		/* === GETTER === */

		/**
		 * Getter for 'show_count' property
		 *
		 * @param string $context What the value is for. Valid values are view and edit.
		 * @return bool
		 */
		public function get_show_count( $context = 'view' ) {
			$show_count = $this->get_prop( 'show_count', $context );

			return ! ! ( is_null( $show_count ) ? 'yes' === self::get_option( 'show_count', 'no' ) : $show_count ) && $this->get_is_single();
		}

		/**
		 * Get the wishlist
		 *
		 * @return false|YITH_WCWL_Wishlist
		 */
		public function get_wishlist() {
			return yith_wcwl_wishlists()->get_wishlist_for_product( $this->get_product_id() );
		}

		/**
		 * Get label for display
		 *
		 * @param string|null $template_kind The kind of template you want the label for
		 *
		 * @return  string
		 */
		public function get_label_for_display( $template_kind = null ) {
			$template_kind = is_null( $template_kind ) ? $this->get_template_kind() : $template_kind;

			return 'move' === $template_kind ? $this->get_option( 'move_label' ) : parent::get_label_for_display( $template_kind );
		}

		/**
		 * Get template kind
		 *
		 * @return string
		 */
		public function get_template_kind() {
			$kind                      = parent::get_template_kind();
			$behaviour                 = self::get_option( 'behaviour', 'view' );
			$add_to_wishlist_modal     = get_option( 'yith_wcwl_modal_enable', 'yes' );
			$multi_wishlist            = yith_wcwl_wishlists()->is_multi_wishlist_enabled();
			$show_popup                = 'default' !== $add_to_wishlist_modal;
			$show_exists_in_a_wishlist = 'add' !== $behaviour;
			$is_in_wishlist            = $this->is_in_a_wishlist();

			$kind = ( 'add' === $behaviour ) ? 'button' : $kind;
			$kind = ( $multi_wishlist && $show_popup && 'yes' === $add_to_wishlist_modal && ( ! isset( $atts['added_to_wishlist'] ) || 'add' === $behaviour ) ) ? 'popup' : $kind;
			$kind = ( 'no' === $add_to_wishlist_modal && $multi_wishlist ) ? 'dropdown' : $kind;
			$kind = ( $multi_wishlist && $show_exists_in_a_wishlist && $is_in_wishlist && 'added' !== $kind ) ? 'browse' : $kind;

			if ( ! ! $is_in_wishlist && in_array( $kind, array( 'browse', 'added' ), true ) ) {
				$kind = 'modal' === $behaviour && $multi_wishlist ? 'move' : $kind;
				$kind = 'remove' === $behaviour ? 'remove' : $kind;
			}

			return $kind;
		}

		/**
		 * Return the array of data used to render the button or false if there is no product recoverable
		 *
		 * @return array|false
		 */
		protected function get_render_data() {
			$data = array(
				'count' => YITH_WCWL_Wishlists::get_instance()->count_add_to_wishlist( $this->get_product_id() ),
			);

			return array_merge( parent::get_render_data(), $data );
		}

		/**
		 * Return the array of data used to render the button or false if there is no product recoverable
		 *
		 * @return array|false
		 */
		protected function get_legacy_php_template_render_data() {
			$data = parent::get_legacy_php_template_render_data();

			if ( ! $data ) {
				return false;
			}

			$move_to_another_wishlist      = 'yes' === self::get_option( 'move_to_another_wishlist' );
			$move_to_another_wishlist_type = self::get_option( 'move_to_another_wishlist_type', 'select' );
			$move_to_another_wishlist_url  = false;
			$item                          = YITH_WCWL_Wishlists::get_instance()->get_product_item( $this->get_product(), 'all' );
			$wishlist                      = $item ? $item->get_wishlist() : false;

			if ( $move_to_another_wishlist && 'popup' === $move_to_another_wishlist_type && $wishlist ) {
				$move_to_another_wishlist_url = esc_url(
					wp_nonce_url(
						add_query_arg(
							'move_to_another_wishlist',
							$wishlist->get_token(),
							$wishlist->get_url()
						),
						'move_to_another_wishlist',
						'move_to_another_wishlist_nonce'
					)
				);
			}

			$additional_data = array(
				'template_part'                 => $this->get_template_kind(),
				'show_exists'                   => 'add' !== self::get_option( 'behaviour' ),
				'show_count'                    => $this->get_show_count(),
				'show_view'                     => $this->get_is_single(),
				'add_to_wishlist_modal'         => self::get_option( 'modal_enabled', 'yes' ),
				'label_popup'                   => $this->get_label_for_display(),
				'available_multi_wishlist'      => yith_wcwl_wishlists()->is_multi_wishlist_enabled(),
				'move_to_another_wishlist'      => $move_to_another_wishlist,
				'move_to_another_wishlist_type' => $move_to_another_wishlist_type,
				'move_to_another_wishlist_url'  => $move_to_another_wishlist_url,
				'link_popup_classes'            => $this->get_popup_button_classes(),
				'disable_wishlist'              => 'yes' === self::get_option( 'disabled_wishlist_for_unauth_users' ),
				'product_image'                 => '',
			);

			if ( $wishlist ) {
				$additional_data['wishlist_url'] = $wishlist->get_url();
			}

			return array_merge( $data, $additional_data );
		}

		/**
		 * Get button classes
		 *
		 * @return string
		 */
		public function get_button_classes() {
			$classes       = parent::get_button_classes();
			$template_part = $this->get_template_kind();

			if ( 'remove' === $template_part ) {
				$classes = str_replace( array( 'single_add_to_wishlist', 'add_to_wishlist' ), '', $classes );
			}

			if ( 'move' === $template_part ) {
				$classes = str_replace( array( 'single_add_to_wishlist', 'add_to_wishlist', 'button' ), '', $classes );
			}
			return $classes;
		}

		/**
		 * Get container HTML classes
		 *
		 * @return string
		 */
		public function get_container_classes() {
			$classes = parent::get_container_classes();

			if ( $this->get_show_count() ) {
				$classes .= ' with-count';
			}

			return $classes;
		}

		/**
		 * Get button classes
		 *
		 * @return string
		 */
		public function get_popup_button_classes() {
			$style   = self::get_option( 'style' );
			$classes = in_array( $style, array( 'button_custom', 'button_default' ), true ) ? array( 'popup_button', 'button', 'alt' ) : array( 'popup_button' );

			if ( 'button_default' === $style ) {
				$classes[] = 'theme-button-style';
				$classes[] = wc_wp_theme_get_element_class_name( 'button' );
			}

			if ( 'popup' === $this->get_template_kind() ) {
				$classes[] = 'add_to_wishlist';
				$classes[] = 'single_add_to_wishlist';
			}

			/**
			 * APPLY_FILTERS: yith_wcwl_add_to_wishlist_popup_classes
			 *
			 * Filter the classes for the 'Add to wishlist' popup.
			 *
			 * @param string $classes Popup classes.
			 *
			 * @return string
			 */
			return apply_filters( 'yith_wcwl_add_to_wishlist_popup_classes', implode( ' ', $classes ) );
		}

		/**
		 * Initialize static options
		 */
		static protected function maybe_init_static_options() {
			static $initialized = false;

			if ( ! $initialized ) {
				static::$options = array_merge(
					static::$options,
					array(
						'move_label' => array(
							'name'    => 'yith_wcwl_add_to_wishlist_move_text',
							'filter'  => 'yith_wcwl_move_from_wishlist_label',
							'default' => __( 'Move to another wishlist', 'yith-woocommerce-wishlist' ),
						),
					)
				);
				$initialized     = true;
			}

			parent::maybe_init_static_options();
		}

		/**
		 * Get object to localize in class script
		 *
		 * @return array
		 */
		protected static function get_data_to_localize() {
			$data            = parent::get_data_to_localize();
			$default_privacy = absint( apply_filters( 'yith_wcwl_wishlist_default_privacy', 0 ) );

			return array_merge_recursive(
				$data,
				array(
					'i18n'                         => array(
						'feedbacks' => apply_filters(
							'yith_wcwl_add_to_wishlist_feedback_modal_labels',
							array(
								'moved' => __( '{{product_name}} has been moved into your {{wishlist_name}} list!', 'yith-woocommerce-wishlist' ),
								'close' => __( 'Close', 'yith-woocommerce-wishlist' ),
							),
						),
						'count'     => array(
							yith_wcwl_get_count_text(),
						),
						'modals'    => array(
							'add_to_wishlist'  => apply_filters(
								'yith_wcwl_add_to_wishlist_modal_labels',
								array(
									'choose_tab'                      => __( 'Choose wishlist', 'yith-woocommerce-wishlist' ),
									'create_tab'                      => __( 'Create a new wishlist', 'yith-woocommerce-wishlist' ),
									'creating_wishlist_button'        => _x( 'Creating the wishlist', '[Modal] The add to wishlist modal button label while creating a new wishlist', 'yith-woocommerce-wishlist' ),
									'adding_wishlist_button'          => _x( 'Adding to the wishlist', '[Modal] The add to wishlist modal button label while adding a product to the wishlist', 'yith-woocommerce-wishlist' ),
									'create_wishlist_button'          => __( 'Create wishlist', 'yith-woocommerce-wishlist' ),
									'wishlist_name_input_placeholder' => __( 'Wishlist name', 'yith-woocommerce-wishlist' ),
								)
							),
							'move_to_wishlist' => apply_filters(
								'yith_wcwl_move_to_wishlist_modal_labels',
								array(
									'title'                       => __( 'Move to another wishlist', 'yith-woocommerce-wishlist' ),
									'description'                 => __( 'This item is already in {{wishlist_name}} list.', 'yith-woocommerce-wishlist' ),
									'move_it_to'                  => __( 'Move it to another list:', 'yith-woocommerce-wishlist' ),
									'choose_wishlist_placeholder' => __( 'Choose a wishlist', 'yith-woocommerce-wishlist' ),
									'move_button'                 => __( 'Move', 'yith-woocommerce-wishlist' ),
									'remove_button'               => __( 'remove it from your list', 'yith-woocommerce-wishlist' ),
								)
							),
						),
					),
					'modalColors'                  => get_option( 'yith_wcwl_modal_colors', yith_wcwl_get_modal_colors_defaults() ),
					'wishlist_url_to_localize'     => YITH_WCWL::get_instance()->get_wishlist_url( 'view/__wishlist_token__' ),
					'default_wishlist_name'        => self::get_option( 'default_wishlist_name' ),
					'is_multi_wishlist'            => YITH_WCWL_Wishlists::get_instance()->is_multi_wishlist_enabled(),
					'privacy_options'              => array_map( 'yith_wcwl_get_privacy_label', array( 0, 1, 2 ) ),
					'privacy_options_descriptions' => array_map( function ( $id ) {
						return yith_wcwl_get_privacy_label( $id, 'only_extension' );
					}, array( 0, 1, 2 ) ),
					'default_privacy'              => in_array( $default_privacy, array( 0, 1, 2 ), true ) ? $default_privacy : 0,
				)
			);
		}
	}
}