<?php
/**
 * Init premium admin features of the plugin
 *
 * @package YITH\Wishlist\Classes
 * @author  YITH <plugins@yithemes.com>
 * @version 3.0.0
 */

if ( ! defined( 'YITH_WCWL' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCWL_Admin_Premium' ) ) {
	/**
	 * WooCommerce Wishlist admin Premium
	 *
	 * @since 1.0.0
	 */
	class YITH_WCWL_Admin_Premium extends YITH_WCWL_Admin_Extended {

		/**
		 * Various links
		 *
		 * @since  1.0.0
		 * @var string
		 * @access public
		 */
		public $showcase_images = array();

		/**
		 * Constructor of the class
		 *
		 * @since 2.0.0
		 */
		public function __construct() {
			parent::__construct();

			// register admin notices.
			add_action( 'admin_notices', array( $this, 'admin_notices' ) );

			// register custom panel handling.
			add_action( 'yith_wcwl_after_popular_table', array( $this, 'print_promotion_wizard' ) );

			// register admin actions.
			add_action( 'admin_action_export_users', array( $this, 'export_users_via_csv' ) );
			add_action( 'admin_action_delete_wishlist', array( $this, 'delete_wishlist_from_actions' ) );
			add_action( 'admin_action_send_promotion', array( $this, 'trigger_promotion_email' ) );

			// adds column to product page.
			add_filter( 'manage_edit-product_columns', array( $this, 'add_product_columns' ) );
			add_filter( 'manage_edit-product_sortable_columns', array( $this, 'product_sortable_columns' ) );
			add_action( 'manage_product_posts_custom_column', array( $this, 'render_product_columns' ) );
			add_filter( 'request', array( $this, 'product_request_query' ) );

			// send promotion email.
			add_action( 'wp_ajax_preview_promotion_email', array( $this, 'ajax_preview_promotion_email' ) );
			add_action( 'wp_ajax_calculate_promotion_email_receivers', array( $this, 'ajax_calculate_promotion_email_receivers' ) );

			// compatibility with email templates.
			add_filter( 'yith_wcet_email_template_types', array( $this, 'register_emails_for_custom_templates' ) );

			// WPML compatibility.
			add_action( 'update_option_yith_wcwl_ask_an_estimate_fields', array( $this, 'register_ask_an_estimate_fields_for_translation' ), 10, 2 );

			// admin only ajax.
			add_action( 'wp_ajax_json_search_coupons', array( $this, 'json_search_coupons' ) );
		}

		/* === INITIALIZATION SECTION === */

		/**
		 * Initiator method. Initiate properties.
		 *
		 * @return void
		 * @access private
		 * @since  1.0.0
		 */
		public function init() {
			parent::init();

			// init scripts needed for the promotion wizard.
			$this->register_promotion_wizard_scripts();
		}

		/**
		 * Register promotion wizard scripts
		 *
		 * @since 3.18.0
		 */
		public function register_promotion_wizard_scripts() {
			$prefix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? 'unminified/' : '';
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			wp_register_script( 'yith-wcwl-promotion-wizard', YITH_WCWL_URL . 'assets/js/' . $prefix . 'admin/yith-wcwl-promotion-wizard' . $suffix . '.js', array( 'jquery', 'wc-backbone-modal', 'jquery-blockui', 'wp-util' ), YITH_WCWL_VERSION, true );

			$emails    = WC()->mailer()->get_emails();
			$email_obj = isset( $emails[ 'yith_wcwl_promotion_mail' ] ) ? $emails[ 'yith_wcwl_promotion_mail' ] : false;

			if ( $email_obj ) {
				$email_obj->init_settings();
				$promotion_settings = $email_obj->settings;
			} else {
				$promotion_settings = get_option( 'woocommerce_yith_wcwl_promotion_mail_settings', array() );
			}

			wp_localize_script(
				'yith-wcwl-admin',
				'yith_wcwl',
				array(
					'promotion' => $promotion_settings,
					'nonce'     => array(
						'preview_promotion_email'             => wp_create_nonce( 'preview_promotion_email' ),
						'calculate_promotion_email_receivers' => wp_create_nonce( 'calculate_promotion_email_receivers' ),
					),
				)
			);
		}

		/* === PANEL HANDLING === */

		/**
		 * Print admin notices for wishlist settings page
		 *
		 * @return void
		 * @since 2.0.7
		 */
		public function admin_notices() {
			$email_sent = isset( $_GET[ 'email_sent' ] ) ? $_GET[ 'email_sent' ] : false; // phpcs:ignore WordPress.Security

			if ( $email_sent ) {
				$res = is_numeric( $email_sent ) ? intval( $email_sent ) : sanitize_text_field( wp_unslash( $email_sent ) );

				if ( $res ) {
					?>
					<div class="updated fade">
						<p><?php esc_html_e( 'Promotional email correctly scheduled', 'yith-woocommerce-wishlist' ); ?></p>
					</div>
					<?php
				} else {
					?>
					<div class="updated fade">
						<p><?php esc_html_e( 'There was an error while scheduling emails; please, try again later', 'yith-woocommerce-wishlist' ); ?></p>
					</div>
					<?php
				}
			}
		}

		/**
		 * Print template for Create Promotion wizard
		 *
		 * @return void
		 * @since 3.0.0
		 */
		public function print_promotion_wizard() {
			$emails    = WC()->mailer()->get_emails();
			$email_obj = isset( $emails[ 'yith_wcwl_promotion_mail' ] ) ? $emails[ 'yith_wcwl_promotion_mail' ] : false;

			if ( ! $email_obj ) {
				return;
			}

			include YITH_WCWL_DIR . 'templates/admin/promotion-wizard.php';
		}

		/* === REQUEST HANDLING === */

		/**
		 * Handle admin requests to delete a wishlist
		 *
		 * @return void
		 * @since 2.0.6
		 */
		public function delete_wishlist_from_actions() {
			if ( ! empty( $_REQUEST[ 'wishlist_id' ] ) ) {
				if ( isset( $_REQUEST[ 'delete_wishlist' ] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST[ 'delete_wishlist' ] ) ), 'delete_wishlist' ) ) {
					$wishlist_id = sanitize_text_field( wp_unslash( $_REQUEST[ 'wishlist_id' ] ) );
					try {
						yith_wcwl_wishlists()->remove( $wishlist_id );
					} catch ( Exception $e ) { // phpcs:ignore
						// do nothing.
					}
				}
			}

			wp_safe_redirect(
				esc_url_raw(
					add_query_arg(
						array(
							'page' => 'yith_wcwl_panel',
							'tab'  => 'dashboard-lists',
						),
						admin_url( 'admin.php' )
					)
				)
			);
			die();
		}

		/**
		 * Export users that added a specific product to their wishlists
		 *
		 * @return void
		 * @since 2.1.3
		 */
		public function export_users_via_csv() {
			if ( ! isset( $_GET[ '_wpnonce' ] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET[ '_wpnonce' ] ) ), 'export_users' ) ) {
				wp_safe_redirect(
					add_query_arg(
						array(
							'page' => 'yith_wcwl_panel',
							'tab'  => 'dashboard-popular',
						),
						admin_url( 'admin.php' )
					)
				);
				die;
			}
			$product_id = isset( $_GET[ 'product_id' ] ) ? intval( $_GET[ 'product_id' ] ) : false;
			$product    = wc_get_product( $product_id );

			$items = YITH_WCWL_Wishlist_Factory::get_wishlist_items(
				array(
					'product_id'  => $product_id,
					'user_id'     => false,
					'session_id'  => false,
					'wishlist_id' => 'all',
				)
			);

			if ( ! empty( $items ) ) {

				$formatted_users = array();

				foreach ( $items as $item ) {
					$user_obj = $item->get_user();
					$user_id  = $item->get_user_id();

					if ( ! $user_obj || isset( $formatted_users[ $user_id ] ) ) {
						continue;
					}

					/**
					 * APPLY_FILTERS: yith_wcwl_csv_export_users_data
					 *
					 * Filter the user data to be exported into CSV.
					 *
					 * @param array   $user_data User data to be exported
					 * @param int     $user_id   User ID
					 * @param WP_User $user_obj  User object
					 *
					 * @return array
					 */
					$formatted_users[ $user_id ] = apply_filters(
						'yith_wcwl_csv_export_users_data',
						array(
							$user_id,
							$user_obj->user_email,
							! empty( $user_obj->billing_first_name ) ? $user_obj->billing_first_name : $user_obj->first_name,
							! empty( $user_obj->billing_last_name ) ? $user_obj->billing_last_name : $user_obj->last_name,
						),
						$user_id,
						$user_obj
					);
				}

				if ( ! empty( $formatted_users ) ) {
					$sitename = sanitize_key( get_bloginfo( 'name' ) );
					$sitename .= ( ! empty( $sitename ) ) ? '-' : '';
					$filename = $sitename . 'wishlist-users-' . sanitize_title_with_dashes( $product->get_title() ) . '-' . gmdate( 'Y-m-d-H-i' ) . '.csv';

					// Add Labels to CSV.
					/**
					 * APPLY_FILTERS: yith_wcwl_csv_export_users_labels
					 *
					 * Filter the labels of the user data to be exported into CSV.
					 *
					 * @param array $labels Array of labels
					 *
					 * @return array
					 */
					$formatted_users_labels[] = apply_filters(
						'yith_wcwl_csv_export_users_labels',
						array(
							__( 'User ID', 'yith-woocommerce-wishlist' ),
							__( 'User Email', 'yith-woocommerce-wishlist' ),
							__( 'User First Name', 'yith-woocommerce-wishlist' ),
							__( 'User Last Name', 'yith-woocommerce-wishlist' ),
						)
					);

					$formatted_users = array_merge( $formatted_users_labels, $formatted_users );

					header( 'Content-Description: File Transfer' );
					header( 'Content-Disposition: attachment; filename=' . $filename );
					header( 'Content-Type: text/xml; charset=' . get_option( 'blog_charset' ), true );

					$df = fopen( 'php://output', 'w' );

					foreach ( $formatted_users as $row ) {
						fputcsv( $df, $row );
					}

					fclose( $df ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
				}
			}

			die();
		}

		/* === WISHLIST COUNT PRODUCT COLUMN === */

		/**
		 * Add column to product table, to show product occurrences in wishlists
		 *
		 * @param array $columns Array of columns for products table.
		 * @return array
		 * @since 2.0.0
		 */
		public function add_product_columns( $columns ) {
			$columns[ 'wishlist_count' ] = __( 'Wishlist Count', 'yith-woocommerce-wishlist' );
			return $columns;
		}

		/**
		 * Render column of occurrences in product table
		 *
		 * @param string $column Column to render.
		 * @return void
		 * @since 2.0.0
		 */
		public function render_product_columns( $column ) {
			global $post;

			if ( 'wishlist_count' === $column ) {
				echo (int) yith_wcwl_wishlists()->count_product_occurrences( $post->ID );
			}
		}

		/**
		 * Register column of occurrences in wishlist as sortable
		 *
		 * @param array $columns Columns that can be sorted in product list table.
		 * @return array
		 * @since 2.0.0
		 */
		public function product_sortable_columns( $columns ) {
			$columns[ 'wishlist_count' ] = 'wishlist_count';
			return $columns;
		}

		/**
		 * Alter post query when ordering for wishlist occurrences
		 *
		 * @param array $vars Arguments used to filter products for the table.
		 * @return array
		 * @since 2.0.0
		 */
		public function product_request_query( $vars ) {
			global $typenow, $wp_query;

			if ( 'product' === $typenow ) {
				// Sorting.
				if ( isset( $vars[ 'orderby' ] ) ) {
					if ( 'wishlist_count' === $vars[ 'orderby' ] ) {
						add_filter( 'posts_join', array( 'YITH_WCWL_Wishlist_Item_Data_Store', 'filter_join_for_wishlist_count' ) );
						add_filter( 'posts_orderby', array( 'YITH_WCWL_Wishlist_Item_Data_Store', 'filter_orderby_for_wishlist_count' ) );
					}
				}
			}

			return $vars;
		}

		/* === SEND PROMOTION EMAIL === */

		/**
		 * Preview promotional email template
		 *
		 * @param bool $return Whether to return or echo the result (@since 3.0.0).
		 *
		 * @return string
		 * @since 2.0.7
		 */
		public function preview_promotion_email( $return = false ) { // phpcs:ignore Universal.NamingConventions.NoReservedKeywordParameterNames.returnFound
			if ( ! isset( $_REQUEST[ '_wpnonce' ] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST[ '_wpnonce' ] ) ), 'preview_promotion_email' ) ) {
				return '';
			}

			$product_id    = isset( $_REQUEST[ 'product_id' ] ) ? array_filter( array_map( 'intval', (array) $_REQUEST[ 'product_id' ] ) ) : false;
			$template      = isset( $_REQUEST[ 'template' ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ 'template' ] ) ) : 'html';
			$content_html  = isset( $_REQUEST[ 'content_html' ] ) ? wp_kses_post( wp_unslash( $_REQUEST[ 'content_html' ] ) ) : false;
			$content_text  = isset( $_REQUEST[ 'content_text' ] ) ? sanitize_textarea_field( wp_unslash( $_REQUEST[ 'content_text' ] ) ) : false;
			$coupon        = isset( $_REQUEST[ 'coupon' ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ 'coupon' ] ) ) : false;
			$template_path = '';

			if ( ! in_array( $template, array( 'html', 'plain' ), true ) ) {
				$template = 'html';
			}

			if ( is_array( $product_id ) ) {
				$product_id = array_shift( $product_id );
			}

			if ( 'plain' === $template ) {
				$template_path = 'plain/';
			}

			// load the mailer class.
			$mailer        = WC()->mailer();
			$email         = $mailer->emails[ 'yith_wcwl_promotion_mail' ];
			$email->user   = get_user_by( 'id', get_current_user_id() );
			$email->object = wc_get_product( $product_id );

			// set contents.
			if ( $content_html ) {
				$email->content_html = wpautop( $content_html );
			}
			if ( $content_text ) {
				$email->content_text = $content_text;
			}

			// set coupon.
			if ( $coupon ) {
				$email->coupon = new WC_Coupon( $coupon );
			}

			// get the preview email subject.
			$email_heading = $email->get_heading();
			$email_content = $email->{'get_custom_content_' . $template}();

			// get the preview email content.
			ob_start();
			include YITH_WCWL_DIR . 'templates/emails/' . $template_path . 'promotion.php';
			$message = ob_get_clean();

			if ( 'plain' === $template ) {
				$message = nl2br( $message );
			}

			$message = $email->style_inline( $message );

			// print the preview email.
			if ( $return ) {
				return $message;
			}

			echo $message; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		/**
		 * Preview promotion email on ajax call
		 *
		 * @return void
		 * @since 3.0.0
		 */
		public function ajax_preview_promotion_email() {
			if ( ! current_user_can( 'manage_woocommerce' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
				die;
			}

			$this->preview_promotion_email();
			die;
		}

		/**
		 * Calculate the number of receivers for the current email and echo it as json content
		 *
		 * @return void
		 */
		public function ajax_calculate_promotion_email_receivers() {
			if ( ! current_user_can( 'manage_woocommerce' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
				die;
			}

			if ( ! isset( $_REQUEST[ '_wpnonce' ] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST[ '_wpnonce' ] ) ), 'calculate_promotion_email_receivers' ) ) {
				die;
			}

			$product_id = isset( $_REQUEST[ 'product_id' ] ) ? array_filter( array_map( 'intval', (array) $_REQUEST[ 'product_id' ] ) ) : false;
			$user_id    = isset( $_REQUEST[ 'user_id' ] ) ? array_filter( array_map( 'intval', (array) $_REQUEST[ 'user_id' ] ) ) : false;

			$count = 0;

			if ( $user_id ) {
				$count = is_array( $user_id ) ? count( $user_id ) : 1;
			} else {
				$receivers_ids = array();

				foreach ( $product_id as $id ) {
					$items = YITH_WCWL_Wishlist_Factory::get_wishlist_items(
						array(
							'wishlist_id' => 'all',
							'session_id'  => false,
							'user_id'     => false,
							'product_id'  => $id,
						)
					);

					if ( ! empty( $items ) ) {
						foreach ( $items as $item ) {
							$receivers_ids[] = $item->get_user_id();
						}
					}

					$receivers_ids = array_unique( $receivers_ids );
					$count         += count( $receivers_ids );
				}
			}

			wp_send_json(
				array(
					'count' => $count,
					'label' => sprintf( '%d %s', $count, _n( 'user', 'users', $count, 'yith-woocommerce-wishlist' ) ),
				)
			);
		}

		/**
		 * Trigger event to send the promotion email
		 *
		 * @return void
		 * @since 2.0.7
		 */
		public function trigger_promotion_email() {
			if ( ! isset( $_POST[ 'send_promotion_email' ] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST[ 'send_promotion_email' ] ) ), 'send_promotion_email_action' ) ) {
				return;
			}

			if ( ! isset( $_POST[ 'product_id' ] ) && ! isset( $_POST[ 'user_id' ] ) ) {
				return;
			}

			$product_id    = isset( $_POST[ 'product_id' ] ) ? $_POST[ 'product_id' ] : false; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
			$user_id       = isset( $_POST[ 'user_id' ] ) ? $_POST[ 'user_id' ] : false; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
			$html_content  = isset( $_POST[ 'content_html' ] ) ? wp_kses_post( wp_unslash( $_POST[ 'content_html' ] ) ) : false;
			$text_content  = isset( $_POST[ 'content_text' ] ) ? sanitize_textarea_field( wp_unslash( $_POST[ 'content_text' ] ) ) : false;
			$coupon_code   = isset( $_POST[ 'coupon' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'coupon' ] ) ) : false;
			$receivers_ids = array();

			$product_id = is_array( $product_id ) ? $product_id : (array) $product_id;
			$product_id = array_filter( array_map( 'intval', $product_id ) );

			$user_id = is_array( $user_id ) ? $user_id : (array) $user_id;
			$user_id = array_filter( array_map( 'intval', $user_id ) );

			// retrieve data about drafts.
			$target = compact( 'product_id', 'user_id' );
			$hash   = md5( http_build_query( $target ) );
			$drafts = get_option( 'yith_wcwl_promotion_drafts', array() );

			// if we're saving draft, update option and skip.
			if ( isset( $_POST[ 'save_draft' ] ) ) {
				$drafts[ $hash ] = array_merge(
					$target,
					array(
						'content_html' => $html_content,
						'content_text' => $text_content,
						'coupon'       => $coupon_code,
					)
				);

				update_option( 'yith_wcwl_promotion_drafts', $drafts );

				wp_safe_redirect(
					esc_url_raw(
						add_query_arg(
							array(
								'page'       => 'yith_wcwl_panel',
								'tab'        => 'dashboard-popular',
								'action'     => $user_id ? 'show_users' : false,
								'product_id' => $user_id ? array_shift( $product_id ) : false,
							),
							admin_url( 'admin.php' )
						)
					)
				);
				exit;
			}

			if ( ! empty( $user_id ) ) {
				$receivers_ids = $user_id;
			} elseif ( ! empty( $product_id ) ) {
				foreach ( $product_id as $id ) {
					$items = YITH_WCWL_Wishlist_Factory::get_wishlist_items(
						array(
							'wishlist_id' => 'all',
							'session_id'  => false,
							'user_id'     => false,
							'product_id'  => $id,
						)
					);

					if ( ! empty( $items ) ) {
						foreach ( $items as $item ) {
							$receivers_ids[] = $item->get_user_id();
						}
					}
				}

				$receivers_ids = array_unique( $receivers_ids );
			}

			if ( ! empty( $receivers_ids ) ) {
				/**
				 * APPLY_FILTERS: yith_wcwl_promotional_email_additional_info
				 *
				 * Filter the additional data required for the Promotional email.
				 *
				 * @param array $data Array of data
				 *
				 * @return array
				 */
				$campaign_info = apply_filters(
					'yith_wcwl_promotional_email_additional_info',
					array(
						'html_content'  => $html_content,
						'text_content'  => $text_content,
						'coupon_code'   => $coupon_code,
						'product_id'    => $product_id,
						'user_id'       => $user_id,
						'receivers'     => $receivers_ids,
						'schedule_date' => time(),
						'counters'      => array(
							'sent'    => 0,
							'to_send' => count( $receivers_ids ),
						),
					)
				);
				// retrieve campaign queue.
				$queue   = get_option( 'yith_wcwl_promotion_campaign_queue', array() );
				$queue[] = $campaign_info;
				$res     = update_option( 'yith_wcwl_promotion_campaign_queue', $queue );
			} else {
				$res = false;
			}

			// finally remove item from drafts.
			if ( isset( $drafts[ $hash ] ) ) {
				unset( $drafts[ $hash ] );

				update_option( 'yith_wcwl_promotion_drafts', $drafts );
			}

			wp_safe_redirect(
				esc_url_raw(
					add_query_arg(
						array(
							'page'       => 'yith_wcwl_panel',
							'tab'        => 'dashboard-popular',
							'email_sent' => ! empty( $res ) ? 'true' : 'false',
							'action'     => $user_id ? 'show_users' : false,
							'product_id' => $user_id ? array_shift( $product_id ) : false,
						),
						admin_url( 'admin.php' )
					)
				)
			);
			exit;
		}

		/* === YITH WOOCOMMERCE EMAIL TEMPLATES INTEGRATION === */

		/**
		 * Filters email template available on yith-wcet
		 *
		 * @param mixed $templates Currently available templates.
		 * @return mixed Fitlered templates
		 * @since 2.0.13
		 */
		public function register_emails_for_custom_templates( $templates ) {
			$templates[] = array(
				'id'   => 'yith-wcwl-ask-an-estimate-mail',
				'name' => __( 'Wishlist "Ask an estimate"', 'yith-woocommerce-wishlist' ),
			);
			$templates[] = array(
				'id'   => 'yith-wcwl-promotion-mail',
				'name' => __( 'Wishlist Promotion', 'yith-woocommerce-wishlist' ),
			);

			return $templates;
		}

		/* === ADMIN ONLY AJAX === */

		/**
		 * Returns coupons upon search
		 *
		 * @param string $term String to match; if nothing is passed, it will be retrieved from query string.
		 * @return void
		 * @since 3.0.0
		 */
		public function json_search_coupons( $term = '' ) {
			if ( ! current_user_can( 'manage_woocommerce' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
				die;
			}

			check_ajax_referer( 'search-products', 'security' );

			if ( empty( $term ) && isset( $_GET[ 'term' ] ) ) {
				$term = (string) sanitize_text_field( wp_unslash( $_GET[ 'term' ] ) );
			}

			if ( empty( $term ) ) {
				wp_die();
			}

			if ( ! empty( $_GET[ 'limit' ] ) ) {
				$limit = absint( $_GET[ 'limit' ] );
			} else {
				$limit = absint( apply_filters( 'woocommerce_json_search_limit', 30 ) );
			}

			$include_ids = ! empty( $_GET[ 'include' ] ) ? array_map( 'absint', (array) wp_unslash( $_GET[ 'include' ] ) ) : array();
			$exclude_ids = ! empty( $_GET[ 'exclude' ] ) ? array_map( 'absint', (array) wp_unslash( $_GET[ 'exclude' ] ) ) : array();

			$coupons        = array();
			$coupon_objects = array();
			$ids            = get_posts(
				array(
					's'              => $term,
					'post_type'      => 'shop_coupon',
					'posts_per_page' => $limit,
					'post__in'       => $include_ids,
					'post__not_id'   => $exclude_ids,
					'fields'         => 'ids',
				)
			);

			if ( ! empty( $ids ) ) {
				foreach ( $ids as $coupon_id ) {
					$coupon_objects[] = new WC_Coupon( $coupon_id );
				}
			}

			foreach ( $coupon_objects as $coupon_object ) {
				$formatted_name = $coupon_object->get_code();

				$coupons[ $formatted_name ] = rawurldecode( $formatted_name );
			}

			wp_send_json( apply_filters( 'woocommerce_json_search_found_coupons', $coupons ) );
		}

		/* === WPML INTEGRATION === */

		/**
		 * Make sure that Ask an Estimate fields are registered for translation on WPML String Translation system
		 *
		 * @param mixed $old_value Previous option value.
		 * @param mixed $new_value Current option value.
		 */
		public function register_ask_an_estimate_fields_for_translation( $old_value, $new_value ) {
			$fields     = yith_wcwl_maybe_format_field_array( $new_value );
			$old_fields = yith_wcwl_maybe_format_field_array( $old_value );
			$old_fields = array_diff_key( $old_fields, $fields );

			if ( $old_fields && function_exists( 'icl_unregister_string' ) ) {
				foreach ( $old_fields as $field_slug => $field ) {
					icl_unregister_string( 'ask-an-estimate-form', "field_{$field_slug}_label" );
					icl_unregister_string( 'ask-an-estimate-form', "field_{$field_slug}_placeholder" );
					icl_unregister_string( 'ask-an-estimate-form', "field_{$field_slug}_description" );
				}
			}

			if ( $fields ) {
				foreach ( $fields as $field_slug => $field ) {
					do_action( 'wpml_register_single_string', 'ask-an-estimate-form', "field_{$field_slug}_label", $field[ 'label' ] );
					do_action( 'wpml_register_single_string', 'ask-an-estimate-form', "field_{$field_slug}_placeholder", $field[ 'placeholder' ] );
					do_action( 'wpml_register_single_string', 'ask-an-estimate-form', "field_{$field_slug}_description", $field[ 'description' ] );
				}
			}
		}
	}
}
