<?php
/**
 * Admin panel premium handler class
 *
 * @package YITH\Wishlist\Admin\Classes
 * @author  YITH <plugins@yithemes.com>
 */

defined( 'YITH_WCWL' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WCWL_Admin_Panel_Premium' ) ) {
	/**
	 * Initiator class. Create and populate admin views.
	 *
	 * @since 4.0.0
	 */
	class YITH_WCWL_Admin_Panel_Premium extends YITH_WCWL_Admin_Panel_Extended {
		/**
		 * YITH_WCWL_Admin_Panel_Premium constructor
		 */
		protected function __construct() {
			parent::__construct();

			// add premium settings.
			add_filter( 'yith_wcwl_settings_options', array( $this, 'add_options_in_general_settings' ) );
			add_filter( 'yith_wcwl_add_to_wishlist_options', array( $this, 'add_options_in_add_to_wishlist_settings' ) );
			add_filter( 'yith_wcwl_promotion_email_options', array( $this, 'add_promotion_email_options' ) );
			add_filter( 'yith_wcwl_customization_labels_options', array( $this, 'add_options_in_customization_labels_settings' ) );
			add_filter( 'yith_wcwl_customization_style_options', array( $this, 'add_options_in_customization_style_settings' ) );
			add_filter( 'yith_wcwl_setting_options_sub_tabs', array( $this, 'add_sub_tabs_in_general_settings' ) );
		}

		/**
		 * Retrieve the admin panel tabs.
		 *
		 * @return array
		 */
		protected function get_tabs(): array {
			$panel_tabs = yith_wcwl_merge_in_array(
				parent::get_tabs(),
				array(
					'email' => array(
						'title' => __( 'Email Settings', 'yith-woocommerce-wishlist' ),
						'icon'  => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6"><path stroke-linecap="round" stroke-linejoin="round" d="M21.75 6.75v10.5a2.25 2.25 0 01-2.25 2.25h-15a2.25 2.25 0 01-2.25-2.25V6.75m19.5 0A2.25 2.25 0 0019.5 4.5h-15a2.25 2.25 0 00-2.25 2.25m19.5 0v.243a2.25 2.25 0 01-1.07 1.916l-7.5 4.615a2.25 2.25 0 01-2.36 0L3.32 8.91a2.25 2.25 0 01-1.07-1.916V6.75" /></svg>',
					),
				),
				'settings'
			);

			$panel_tabs[ 'dashboard' ][ 'title' ] = _x( 'Dashboard', 'Settings tab name', 'yith-woocommerce-wishlist' );
			unset( $panel_tabs[ 'dashboard' ][ 'desc' ] );

			return $panel_tabs;
		}


		/**
		 * Add new options to general settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_options_in_general_settings( $options ) {
			return $this->add_premium_options( 'settings-general', $options );
		}

		/**
		 * Add new options to Add to Wishlist settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_options_in_add_to_wishlist_settings( $options ) {
			$options = $this->add_premium_options( 'settings-add_to_wishlist', $options );

			if ( 'yes' === get_option( 'yith_wcwl_multi_wishlist_enable', 'yes' ) ) {
				$options[ 'settings-add_to_wishlist' ][ 'after_add_to_wishlist_behaviour' ][ 'options' ][ 'modal' ] = __( 'Show a modal window with move/remove options (available only if the multi-wishlist option is enabled).', 'yith-woocommerce-wishlist' );
			}

			return $options;
		}

		/**
		 * Add new options to wishlist settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_options_in_wishlist_page_settings( $options ) {
			$options = parent::add_options_in_wishlist_page_settings( $options );

			return $this->add_premium_options( 'settings-wishlist_page', $options );
		}

		/**
		 * Add new options to Promotional settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_promotion_email_options( $options ) {
			return $this->add_premium_options( 'promotion_email', $options );
		}

		/**
		 * Add new options to Customization labels settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_options_in_customization_labels_settings( $options ) {
			return $this->add_premium_options( 'customization-labels', $options );
		}

		/**
		 * Add new options to Customization style settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_options_in_customization_style_settings( $options ) {
			return $this->add_premium_options( 'customization-style', $options );
		}

		/**
		 * Add premium sub-tabs into the general settings tab
		 *
		 * @param array[] $sub_tabs Array of sub-tabs and their details.
		 *
		 * @return array
		 */
		public function add_sub_tabs_in_general_settings( $sub_tabs ) {
			$premium_sub_tabs = array(
				'settings-ask_an_estimate' => array(
					'title'              => _x( 'Ask for an estimate', 'Tab title in plugin settings panel', 'yith-woocommerce-wishlist' ),
					'yith-wcwl-priority' => 20,
					'description'        => _x( 'Configure the Ask an estimate settings.', 'Tab description in plugin settings panel', 'yith-woocommerce-wishlist' ),
				),
			);

			$sub_tabs = yith_wcwl_merge_in_array( $sub_tabs, $premium_sub_tabs, 'settings-wishlist_page' );

			return $sub_tabs;
		}

		/**
		 * Add the premium options to the panel tab settings
		 *
		 * @param string $tab     The panel tab name
		 * @param array  $options The settings options array
		 *
		 * @return array
		 */
		private function add_premium_options( $tab, $options ) {
			$path             = YITH_WCWL_DIR . 'plugin-options/premium-options/' . str_replace( '-', '/', $tab ) . '-options.php';
			$premium_settings = file_exists( $path ) ? include $path : array();

			$settings = $options[ $tab ];
			foreach ( $premium_settings as $premium_options ) {
				if ( array_key_exists( 'options', $premium_options ) && is_array( $premium_options[ 'options' ] ) ) {
					if ( ! empty( $premium_options[ 'pivot' ] ) && array_key_exists( $premium_options[ 'pivot' ], $settings ) ) {
						$settings = yith_wcwl_merge_in_array(
							$settings,
							$premium_options[ 'options' ],
							$premium_options[ 'pivot' ],
							$premium_options[ 'position' ] ?? 'after'
						);
					} else {
						$settings = array_merge( $settings, $premium_options[ 'options' ] );
					}
				}
			}

			$options[ $tab ] = $settings;

			return $options;
		}

		/**
		 * Return false as it is already PREMIUM! :)
		 *
		 * @return false
		 */
		protected function get_premium_tab() {
			return array();
		}

		/**
		 * Get 'Your store tools tab' options
		 *
		 * @return array[]
		 */
		protected function get_your_store_tools_tab() {
			return array(
				'items' => array(
					'gift-cards'             => array(
						'name'           => 'Gift Cards',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/gift-cards.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-gift-cards/',
						'description'    => _x(
							'Sell gift cards in your shop to increase your earnings and attract new customers.',
							'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Gift Cards',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_YWGC_PREMIUM' ),
						'is_recommended' => true,
					),
					'ajax-product-filter'    => array(
						'name'           => 'Ajax Product Filter',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/ajax-product-filter.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-ajax-product-filter/',
						'description'    => _x(
							'Help your customers to easily find the products they are looking for and improve the user experience of your shop.',
							'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Ajax Product Filter',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_WCAN_PREMIUM' ),
						'is_recommended' => true,
					),
					'booking'                => array(
						'name'           => 'Booking and Appointment',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/booking.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-booking/',
						'description'    => _x(
							'Enable a booking/appointment system to manage renting or booking of services, rooms, houses, cars, accommodation facilities and so on.',
							'[YOUR STORE TOOLS TAB] Description for plugin YITH Bookings',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_WCBK_PREMIUM' ),
						'is_recommended' => false,

					),
					'request-a-quote'        => array(
						'name'           => 'Request a Quote',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/request-a-quote.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-request-a-quote/',
						'description'    => _x(
							'Hide prices and/or the "Add to cart" button and let your customers request a custom quote for every product.',
							'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Request a Quote',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_YWRAQ_PREMIUM' ),
						'is_recommended' => false,
					),
					'product-addons'         => array(
						'name'           => 'Product Add-Ons & Extra Options',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/product-add-ons.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-product-add-ons/',
						'description'    => _x(
							'Add paid or free advanced options to your product pages using fields like radio buttons, checkboxes, drop-downs, custom text inputs, and more.',
							'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Product Add-Ons',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_WAPO_PREMIUM' ),
						'is_recommended' => false,
					),
					'dynamic-pricing'        => array(
						'name'           => 'Dynamic Pricing and Discounts',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/dynamic-pricing-and-discounts.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-dynamic-pricing-and-discounts/',
						'description'    => _x(
							'Increase conversions through dynamic discounts and price rules, and build powerful and targeted offers.',
							'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Dynamic Pricing and Discounts',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_YWDPD_PREMIUM' ),
						'is_recommended' => false,
					),
					'customize-my-account'   => array(
						'name'           => 'Customize My Account Page',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/customize-myaccount-page.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-customize-my-account-page/',
						'description'    => _x( 'Customize the My Account page of your customers by creating custom sections with promotions and ad-hoc content based on your needs.', '[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Customize My Account', 'yith-woocommerce-wishlist' ),
						'is_active'      => defined( 'YITH_WCMAP_PREMIUM' ),
						'is_recommended' => false,
					),
					'recover-abandoned-cart' => array(
						'name'           => 'Recover Abandoned Cart',
						'icon_url'       => YITH_WCWL_URL . 'assets/images/plugins/recover-abandoned-cart.svg',
						'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-recover-abandoned-cart/',
						'description'    => _x(
							'Contact users who have added products to the cart without completing the order and try to recover lost sales.',
							'[YOUR STORE TOOLS TAB] Description for plugin Recover Abandoned Cart',
							'yith-woocommerce-wishlist'
						),
						'is_active'      => defined( 'YITH_YWRAC_PREMIUM' ),
						'is_recommended' => false,
					),
				),
			);
		}
	}
}

