<?php
/**
 * Admin panel handler class
 *
 * @package YITH\Wishlist\Admin\Classes
 * @author  YITH <plugins@yithemes.com>
 */

defined( 'YITH_WCWL' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WCWL_Admin_Panel_Extended' ) ) {
	/**
	 * Initiator class. Create and populate admin views.
	 *
	 * @since 4.0.0
	 */
	class YITH_WCWL_Admin_Panel_Extended extends YITH_WCWL_Admin_Panel {
		protected function __construct() {
			parent::__construct();

			// add premium settings.
			add_filter( 'yith_wcwl_wishlist_page_options', array( $this, 'add_options_in_wishlist_page_settings' ) );

			// email settings.
			add_action( 'yith_wcwl_email_settings', array( $this, 'print_email_settings_tab' ) );
			add_action( 'yith_wcwl_print_email_settings', array( $this, 'print_email_settings' ) );
		}

		/**
		 * Retrieve the admin panel tabs.
		 *
		 * @return array
		 */
		protected function get_tabs(): array {
			$new_tabs = array(
				'dashboard' => array(
					'title'       => _x( 'Popular', 'Settings tab name', 'yith-woocommerce-wishlist' ),
					'description' => _x( 'Check the most popular products in your site.', 'Tab description in plugin settings panel', 'yith-woocommerce-wishlist' ),
					'icon'        => 'dashboard',
				),
				'email'     => array(
					'title' => __( 'Email Settings', 'yith-woocommerce-wishlist' ),
					'icon'  => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6"><path stroke-linecap="round" stroke-linejoin="round" d="M21.75 6.75v10.5a2.25 2.25 0 01-2.25 2.25h-15a2.25 2.25 0 01-2.25-2.25V6.75m19.5 0A2.25 2.25 0 0019.5 4.5h-15a2.25 2.25 0 00-2.25 2.25m19.5 0v.243a2.25 2.25 0 01-1.07 1.916l-7.5 4.615a2.25 2.25 0 01-2.36 0L3.32 8.91a2.25 2.25 0 01-1.07-1.916V6.75" /></svg>',
				),
			);

			$panel_tabs = yith_wcwl_merge_in_array( parent::get_tabs(), array( 'dashboard' => $new_tabs[ 'dashboard' ] ), 'settings', 'before' );
			$panel_tabs = yith_wcwl_merge_in_array( $panel_tabs, array( 'email' => $new_tabs[ 'email' ] ), 'settings' );

			return $panel_tabs;
		}


		/**
		 * Add new options to wishlist settings tab
		 *
		 * @param array $options Array of available options.
		 * @return array Filtered array of options
		 */
		public function add_options_in_wishlist_page_settings( $options ) {
			$options[ 'settings-wishlist_page' ] = yith_wcwl_merge_in_array(
				$options[ 'settings-wishlist_page' ],
				array(
					'show_quantity' => array(
						'name'          => __( 'In wishlist table show', 'yith-woocommerce-wishlist' ),
						'desc'          => __( 'Product quantity (so users can manage the quantity of each product from the wishlist)', 'yith-woocommerce-wishlist' ),
						'id'            => 'yith_wcwl_quantity_show',
						'type'          => 'checkbox',
						'default'       => '',
						'checkboxgroup' => 'wishlist_info',
					),
				),
				'show_unit_price'
			);

			return $options;
		}

		/**
		 * Handle email settings tab
		 * This method based on query string load single email options or the general table
		 */
		public function print_email_settings_tab() {
			$emails = YITH_WCWL()->emails;

			// load mailer.
			$mailer = WC()->mailer();

			$emails_table = array();
			foreach ( $emails as $email ) {

				$email_class            = $mailer->emails[ $email ];
				$emails_table[ $email ] = array(
					'title'       => $email_class->get_title(),
					'description' => $email_class->get_description(),
					'recipient'   => $email_class->is_customer_email() ? __( 'Customer', 'yith-woocommerce-gift-cards' ) : $email_class->get_recipient(),
					'enable'      => $email_class->is_enabled(),
					'content'     => $email_class->get_content_type(),
				);
			}

			yith_wcwl_get_template( 'admin/email-settings-tab.php', compact( 'emails_table' ) );
		}

		/**
		 * Output email settings section
		 *
		 * @param string $email_key Email ID.
		 *
		 * @return void
		 */
		public function print_email_settings( $email_key ) {
			global $current_section;
			$current_section = strtolower( $email_key );
			$mailer          = WC()->mailer();
			$class           = $mailer->emails[ $email_key ];
			WC_Admin_Settings::get_settings_pages();

			if ( ! empty( $_POST ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
				$class->process_admin_options();
			}

			yith_wcwl_get_template( 'admin/email-settings-single.php' );

			$current_section = null;
		}

		/**
		 * Save email settings in ajax.
		 *
		 * @return void
		 */
		public function save_email_settings() {
			// phpcs:disable WordPress.Security.NonceVerification.Missing
			if ( isset( $_POST[ 'params' ] ) ) {
				parse_str( $_POST[ 'params' ], $params ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				unset( $_POST[ 'params' ] );

				foreach ( $params as $key => $value ) {
					$_POST[ $key ] = $value;
				}

				global $current_section;

				$email_key       = isset( $_POST[ 'email_key' ] ) ? strtolower( sanitize_text_field( wp_unslash( $_POST[ 'email_key' ] ) ) ) : '';
				$current_section = $email_key;

				$mailer = WC()->mailer();
				$class  = $mailer->emails[ $email_key ];
				$class->process_admin_options();

				$current_section = null;

				wp_send_json_success( array( 'msg' => 'Email updated' ) );
				die();
			}
			// phpcs:enable WordPress.Security.NonceVerification.Missing
		}
	}
}

